/*****************************************************************************
*                           Freescale Semiconductor                          *

*                                                                            *
*  Project    : M56F8006 Modular Pixel Matrix                                                  *
*  Version    : 0.1                                                          *
*  Date       : 11/Apr/2009                                                  *
*  Authors     : Alexandre Dias, Bruno Bastos                                 *
*               Humberto Martins, Leonardo Mangiapelo                        *
*               Renato Frias                                                 *
*                                                                            *
* -------------------------------------------------------------------------- *
*  Description:                                                              *
*                                                                            *
*    M56F8006 Modular Pixel Matrix												                                     * 
*                                                                            *
******************************************************************************/

#ifndef __MCD_C_API_H   
#define __MCD_C_API_H		1


///////////////////////////////////////////////////////////////////////////////
// T Y P E   D E F I N I T I O N S
///////////////////////////////////////////////////////////////////////////////

// Address/constant definitions for this SoC
typedef uint8_t                 command_t;
typedef uint32_t                address_t;

// Testbench communication for this SoC
typedef uint8_t                 flag_t;
typedef uint8_t                 flag_status_t;
typedef uint8_t                 mailbox_t;
typedef uint8_t                 mailbox_status_t;


///////////////////////////////////////////////////////////////////////////////
// G L O B A L   V A R I A B L E S
///////////////////////////////////////////////////////////////////////////////
#pragma define_section NOINIT ".noinit" RW

// C-Print variable space
extern uint32_t cprint_cmd;
extern uint32_t cprint_arg[5];
extern uint8_t  noinit8[8];
extern uint16_t noinit16[8];
extern uint32_t noinit32[8];

// Pass/fail notification implementation for this SoC
extern uint32_t error_counter;
extern uint8_t  fail_status;


////////////////////////////////////////////////////////////////////////////////
// F U N C T I O N   P R O T O T Y P E S
////////////////////////////////////////////////////////////////////////////////

// Testbench notification implementation for this SoC
extern void _notify_tb_caa(command_t command, address_t address, address_t address2);
extern void _notify_tb_caa(command_t command, address_t address, address_t address2);
extern void _notify_tb_ca8(command_t command, address_t address, uint8_t data1);
extern void _notify_tb_ca16(command_t command, address_t address, uint16_t data1);
extern void _notify_tb_ca32(command_t command, address_t address, uint32_t data1);
extern void _notify_tb_caa8(command_t command, address_t address, address_t address2, uint8_t data1);
extern void _notify_tb_caa16(command_t command, address_t address, address_t address2, uint16_t data1);
extern void _notify_tb_caa32(command_t command, address_t address, address_t address2, uint32_t data1);
extern void _notify_tb_ca88(command_t command, address_t address, uint8_t data1, uint8_t data2);
extern void _notify_tb_ca1616(command_t command, address_t address, uint16_t data1, uint16_t data2);
extern void _notify_tb_ca3232(command_t command, address_t address, uint32_t data1, uint32_t data2);
extern void _notify_tb_ca888(command_t command, address_t address, uint8_t data1, uint8_t data2, uint8_t data3);
extern void _notify_tb_ca161616(command_t command, address_t address, uint16_t data1, uint16_t data2, uint16_t data3);
extern void _notify_tb_ca323232(command_t command, address_t address, uint32_t data1, uint32_t data2, uint32_t data3);
extern void _notify_tb_ca8816(command_t command, address_t address, uint8_t data1, uint8_t data2, uint16_t data3);
extern void _notify_tb_ca323216(command_t command, address_t address, uint32_t data1, uint32_t data2, uint16_t data3);
extern void _notify_tb_ca8888(command_t command, address_t address, uint8_t data1, uint8_t data2, uint8_t data3, uint8_t data4);
extern void _notify_tb_ca16161616(command_t command, address_t address, uint16_t data1, uint16_t data2, uint16_t data3, uint16_t data4);
extern void _notify_tb_ca32323232(command_t command, address_t address, uint32_t data1, uint32_t data2, uint32_t data3, uint32_t data4);
extern void _notify_tb_ca88816(command_t command, address_t address, uint8_t data1, uint8_t data2, uint8_t data3, uint16_t data4);
extern void _notify_tb_ca32323216(command_t command, address_t address, uint32_t data1, uint32_t data2, uint32_t data3, uint16_t data4);

// Pass/fail notification implementation for this SoC
extern void _fail_exit(void);
extern void _success(void);
extern void _failure(void);
extern void _terminate(void);
extern void _trigger_set(flag_t flag);
extern void _trigger_wait(flag_t flag);
extern void _mb_put8(mailbox_t mailbox, uint8_t write_data);
extern void _mb_put16(mailbox_t mailbox, uint16_t write_data);
extern void _mb_put32(mailbox_t mailbox, uint32_t write_data);
extern void _mb_get8(mailbox_t mailbox, uint8_t* destvar);
extern void _mb_get16(mailbox_t mailbox, uint16_t* destvar);
extern void _mb_get32(mailbox_t mailbox, uint32_t* destvar);

// MCD C API implementation for this SoC
extern void _r8(vuint8_t* address, uint8_t* destvar);
extern void _r16(vuint16_t* address, uint16_t* destvar);
extern void _r32(vuint32_t* address, uint32_t* destvar);
extern void _re8(vuint8_t* address, uint8_t expect_data);
extern void _re16(vuint16_t* address, uint16_t expect_data);
extern void _re32(vuint32_t* address, uint32_t expect_data);
extern void _rem8(vuint8_t* address, uint8_t expect_data, uint8_t mask);
extern void _rem16(vuint16_t* address, uint16_t expect_data, uint16_t mask);
extern void _rem32(vuint32_t* address, uint32_t expect_data, uint32_t mask);
extern void _rre8(vuint8_t* address, vuint8_t* destvar, uint8_t expect_data);
extern void _rre16(vuint16_t* address, vuint16_t* destvar, uint16_t expect_data);
extern void _rre32(vuint32_t* address, vuint32_t* destvar, uint32_t expect_data);
extern void _rrem8(vuint8_t* address, vuint8_t* destvar, uint8_t expect_data, uint8_t mask);
extern void _rrem16(vuint16_t* address, vuint16_t* destvar, uint16_t expect_data, uint16_t mask);
extern void _rrem32(vuint32_t* address, vuint32_t* destvar, uint32_t expect_data, uint32_t mask);
extern void _w8(vuint8_t* address, uint8_t write_data);
extern void _w16(vuint16_t* address, uint16_t write_data);
extern void _w32(vuint32_t* address, uint32_t write_data);
extern void _wr8(vuint8_t* address, uint8_t write_data);
extern void _wr16(vuint16_t* address, uint16_t write_data);
extern void _wr32(vuint32_t* address, uint32_t write_data);
extern void _wrm8(vuint8_t* address, uint8_t write_data, uint8_t mask);
extern void _wrm16(vuint16_t* address, uint16_t write_data, uint16_t mask);
extern void _wrm32(vuint32_t* address, uint32_t write_data, uint32_t mask);
extern void _wre8(vuint8_t* address, uint8_t write_data, uint8_t expect_data);
extern void _wre16(vuint16_t* address, uint16_t write_data, uint16_t expect_data);
extern void _wre32(vuint32_t* address, uint32_t write_data, uint32_t expect_data);
extern void _wrem8(vuint8_t* address, uint8_t write_data, uint8_t expect_data, uint8_t mask);
extern void _wrem16(vuint16_t* address, uint16_t write_data, uint16_t expect_data, uint16_t mask);
extern void _wrem32(vuint32_t* address, uint32_t write_data, uint32_t expect_data, uint32_t mask);
extern void _wait_re8(vuint8_t* address, uint8_t expect_data, uint16_t read_attempts);
extern void _wait_re16(vuint16_t* address, uint16_t expect_data, uint16_t read_attempts);
extern void _wait_re32(vuint32_t* address, uint32_t expect_data, uint16_t read_attempts);
extern void _wait_rem8(vuint8_t* address, uint8_t expect_data, uint8_t mask, uint16_t read_attempts);
extern void _wait_rem16(vuint16_t* address, uint16_t expect_data, uint16_t mask, uint16_t read_attempts);
extern void _wait_rem32(vuint32_t* address, uint32_t expect_data, uint32_t mask, uint16_t read_attempts);
extern void _bset8(vuint8_t* address, uint8_t mask);
extern void _bset16(vuint16_t* address, uint16_t mask);
extern void _bset32(vuint32_t* address, uint32_t mask);
extern void _bclr8(vuint8_t* address, uint8_t mask);
extern void _bclr16(vuint16_t* address, uint16_t mask);
extern void _bclr32(vuint32_t* address, uint32_t mask);
extern void _binv8(vuint8_t* address, uint8_t mask);
extern void _binv16(vuint16_t* address, uint16_t mask);
extern void _binv32(vuint32_t* address, uint32_t mask);
extern void _debug(char* source, char* comment);
extern void _trace(char* source, char* comment);
extern void _info(char* source, char* comment);
extern void _warn(char* source, char* comment);
extern void _error(char* source, char* comment);
extern void _fatal(char* source, char* comment);
extern void _show8(char* source, char* comment, uint8_t value);
extern void _show16(char* source, char* comment, uint16_t value);
extern void _show32(char* source, char* comment, uint32_t value);


///////////////////////////////////////////////////////////////////////////////
// C - P R I N T   M A C R O S
///////////////////////////////////////////////////////////////////////////////

// Use the following to select appropriate typedefs in generic macros
// Note: default is 8-bit data type
#define uint_t(x)       uint##x##_t
#define vuint_t(x)      vuint##x##_t

// Address/constant definitions for this SoC
#ifndef CPRINT_BASE
#define CPRINT_BASE     0x03D6
#endif
#define CPRINT_CMD      (*(vuint32_t*)(CPRINT_BASE+0x00))
#define CPRINT_ARG0     (*(vuint32_t*)(CPRINT_BASE+0x02))
#define CPRINT_ARG1     (*(vuint32_t*)(CPRINT_BASE+0x04))
#define CPRINT_ARG2     (*(vuint32_t*)(CPRINT_BASE+0x06))
#define CPRINT_ARG3     (*(vuint32_t*)(CPRINT_BASE+0x08))
#define CPRINT_ARG4     (*(vuint32_t*)(CPRINT_BASE+0x0A))
#define EXIT_PORTAL     (*(vuint32_t*)(CPRINT_BASE+0x0C))

// C-Print Function Macros
#define CPRINT_R8                0x00
#define CPRINT_R16               0x01
#define CPRINT_R32               0x02
#define CPRINT_S_RE8             0x03
#define CPRINT_S_RE16            0x04
#define CPRINT_S_RE32            0x05
#define CPRINT_S_REM8            0x06
#define CPRINT_S_REM16           0x07   
#define CPRINT_S_REM32           0x08   
#define CPRINT_S_RRE8            0x09   
#define CPRINT_S_RRE16           0x0A   
#define CPRINT_S_RRE32           0x0B   
#define CPRINT_S_RREM8           0x0C   
#define CPRINT_S_RREM16          0x0D   
#define CPRINT_S_RREM32          0x0E   
#define CPRINT_E_RE8             0x0F   
#define CPRINT_E_RE16            0x10   
#define CPRINT_E_RE32            0x11   
#define CPRINT_E_REM8            0x12   
#define CPRINT_E_REM16           0x13   
#define CPRINT_E_REM32           0x14   
#define CPRINT_E_RRE8            0x15   
#define CPRINT_E_RRE16           0x16   
#define CPRINT_E_RRE32           0x17   
#define CPRINT_E_RREM8           0x18   
#define CPRINT_E_RREM16          0x19   
#define CPRINT_E_RREM32          0x1A   
#define CPRINT_W8                0x1B   
#define CPRINT_W16               0x1C   
#define CPRINT_W32               0x1D      
#define CPRINT_S_WR8             0x1E      
#define CPRINT_S_WR16            0x1F      
#define CPRINT_S_WR32            0x20      
#define CPRINT_S_WRM8            0x21      
#define CPRINT_S_WRM16           0x22      
#define CPRINT_S_WRM32           0x23      
#define CPRINT_S_WRE8            0x24      
#define CPRINT_S_WRE16           0x25      
#define CPRINT_S_WRE32           0x26      
#define CPRINT_S_WREM8           0x27      
#define CPRINT_S_WREM16          0x28      
#define CPRINT_S_WREM32          0x29      
#define CPRINT_E_WR8             0x2A      
#define CPRINT_E_WR16            0x2B      
#define CPRINT_E_WR32            0x2C      
#define CPRINT_E_WRM8            0x2D      
#define CPRINT_E_WRM16           0x2E      
#define CPRINT_E_WRM32           0x2F      
#define CPRINT_E_WRE8            0x30      
#define CPRINT_E_WRE16           0x31      
#define CPRINT_E_WRE32           0x32      
#define CPRINT_E_WREM8           0x33      
#define CPRINT_E_WREM16          0x34      
#define CPRINT_E_WREM32          0x35      
#define CPRINT_S_WAIT_RE8        0x36      
#define CPRINT_S_WAIT_RE16       0x37      
#define CPRINT_S_WAIT_RE32       0x38      
#define CPRINT_S_WAIT_REM8       0x39      
#define CPRINT_S_WAIT_REM16      0x3A      
#define CPRINT_S_WAIT_REM32      0x3B      
#define CPRINT_E_WAIT_RE8        0x3C      
#define CPRINT_E_WAIT_RE16       0x3D      
#define CPRINT_E_WAIT_RE32       0x3E      
#define CPRINT_E_WAIT_REM8       0x3F      
#define CPRINT_E_WAIT_REM16      0x40      
#define CPRINT_E_WAIT_REM32      0x41      
#define CPRINT_BCLR8             0x42      
#define CPRINT_BCLR16            0x43      
#define CPRINT_BCLR32            0x44      
#define CPRINT_BINV8             0x45      
#define CPRINT_BINV16            0x46      
#define CPRINT_BINV32            0x47      
#define CPRINT_BSET8             0x48      
#define CPRINT_BSET16            0x49      
#define CPRINT_BSET32            0x4A      
#define CPRINT_DEBUG             0x4B      
#define CPRINT_TRACE             0x4C      
#define CPRINT_INFO              0x4D      
#define CPRINT_WARN              0x4E      
#define CPRINT_ERROR             0x4F      
#define CPRINT_FATAL             0x50      
#define CPRINT_TRIGGER_SET       0x51      
#define CPRINT_TRIGGER_WAIT      0x52      
#define CPRINT_MB_PUT8           0x53      
#define CPRINT_MB_PUT16          0x54      
#define CPRINT_MB_PUT32          0x55      
#define CPRINT_MB_GET8           0x56      
#define CPRINT_MB_GET16          0x57      
#define CPRINT_MB_GET32          0x58      
#define CPRINT_SHOW8             0x59      
#define CPRINT_SHOW16            0x5A      
#define CPRINT_SHOW32            0x5B      

// C-Print Command Macros
#define CCMDx1(a,x) { CPRINT_ARG0=(a); CPRINT_CMD=(x); }
#define CCMDx2(a,b,x) { CPRINT_ARG0=(a); CPRINT_ARG1=(b); CPRINT_CMD=(x); }
#define CCMDx3(a,b,c,x) { CPRINT_ARG0=(a); CPRINT_ARG1=(b); CPRINT_ARG2=(c); CPRINT_CMD=(x); }
#define CCMDx4(a,b,c,d,x) { CPRINT_ARG0=(a); CPRINT_ARG1=(b); CPRINT_ARG2=(c); CPRINT_ARG3=(d); CPRINT_CMD=(x); }
#define CCMDx5(a,b,c,d,e,x) { CPRINT_ARG0=(a); CPRINT_ARG1=(b); CPRINT_ARG2=(c); CPRINT_ARG3=(d); CPRINT_ARG4=(e); CPRINT_CMD=(x); }


///////////////////////////////////////////////////////////////////////////////
// Tesbench notification for this SoC
///////////////////////////////////////////////////////////////////////////////

#define _NOTIFY_TB_CAA(command, address, address2) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (address_t)(address2); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA8(command, address, data1) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint8_t)(data1); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA16(command, address, data1) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint16_t)(data1); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA32(command, address, data1) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint32_t)(data1); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CAA8(command, address, address2, data1) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (address_t)(address2); \
  CPRINT_ARG2 = (uint8_t)(data1); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CAA16(command, address, address2, data1) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (address_t)(address2); \
  CPRINT_ARG2 = (uint16_t)(data1); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CAA32(command, address, address2, data1) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (address_t)(address2); \
  CPRINT_ARG2 = (uint32_t)(data1); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA88(command, address, data1, data2) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint8_t)(data1); \
  CPRINT_ARG2 = (uint8_t)(data2); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA1616(command, address, data1, data2) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint16_t)(data1); \
  CPRINT_ARG2 = (uint16_t)(data2); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA3232(command, address, data1, data2) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint32_t)(data1); \
  CPRINT_ARG2 = (uint32_t)(data2); \
  CPRINT_ARG2 = (uint32_t)(data2); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA888(command, address, data1, data2, data3) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint8_t)(data1); \
  CPRINT_ARG2 = (uint8_t)(data2); \
  CPRINT_ARG3 = (uint8_t)(data3); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA161616(command, address, data1, data2, data3) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint16_t)(data1); \
  CPRINT_ARG2 = (uint16_t)(data2); \
  CPRINT_ARG3 = (uint16_t)(data3); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA323232(command, address, data1, data2, data3) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint32_t)(data1); \
  CPRINT_ARG2 = (uint32_t)(data2); \
  CPRINT_ARG3 = (uint32_t)(data3); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA8816(command, address, data1, data2, data3) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint8_t)(data1); \
  CPRINT_ARG2 = (uint8_t)(data2); \
  CPRINT_ARG3 = (uint16_t)(data3); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA323216(command, address, data1, data2, data3) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint32_t)(data1); \
  CPRINT_ARG2 = (uint32_t)(data2); \
  CPRINT_ARG3 = (uint16_t)(data3); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA8888(command, address, data1, data2, data3, data4) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint8_t)(data1); \
  CPRINT_ARG2 = (uint8_t)(data2); \
  CPRINT_ARG3 = (uint8_t)(data3); \
  CPRINT_ARG4 = (uint8_t)(data4); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA16161616(command, address, data1, data2, data3, data4) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint16_t)(data1); \
  CPRINT_ARG2 = (uint16_t)(data2); \
  CPRINT_ARG3 = (uint16_t)(data3); \
  CPRINT_ARG4 = (uint16_t)(data4); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA32323232(command, address, data1, data2, data3, data4) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint32_t)(data1); \
  CPRINT_ARG2 = (uint32_t)(data2); \
  CPRINT_ARG3 = (uint32_t)(data3); \
  CPRINT_ARG4 = (uint32_t)(data4); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA88816(command, address, data1, data2, data3, data4) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint8_t)(data1); \
  CPRINT_ARG2 = (uint8_t)(data2); \
  CPRINT_ARG3 = (uint8_t)(data3); \
  CPRINT_ARG4 = (uint16_t)(data4); \
  CPRINT_CMD  = (command_t)(command); \
}

#define _NOTIFY_TB_CA32323216(command, address, data1, data2, data3, data4) { \
  CPRINT_ARG0 = (address_t)(address); \
  CPRINT_ARG1 = (uint32_t)(data1); \
  CPRINT_ARG2 = (uint32_t)(data2); \
  CPRINT_ARG3 = (uint32_t)(data3); \
  CPRINT_ARG4 = (uint16_t)(data4); \
  CPRINT_CMD  = (command_t)(command); \
}

////////////////////////////////////////////////////////////////////////////////
// T E S T B E N C H   N O T I F I C A T I O N
////////////////////////////////////////////////////////////////////////////////

// Tesbench notification for this SoC (function implementation)
#if (MCD_TB_NOTIFY == _USE_FUNC)
#define NOTIFY_TB_CAA(a,b,c)               _notify_tb_caa((command_t)a,(address_t)b,(address_t)c)
#define NOTIFY_TB_CA8(a,b,c)               _notify_tb_ca8((command_t)a,(address_t)b,(uint8_t)c)
#define NOTIFY_TB_CA16(a,b,c)              _notify_tb_ca16((command_t)a,(address_t)b,(uint16_t)c)
#define NOTIFY_TB_CA32(a,b,c)              _notify_tb_ca32((command_t)a,(address_t)b,(uint32_t)c)
#define NOTIFY_TB_CAA8(a,b,c,d)            _notify_tb_caa8((command_t)a,(address_t)b,(address_t)c,(uint8_t)d)
#define NOTIFY_TB_CAA16(a,b,c,d)           _notify_tb_caa16((command_t)a,(address_t)b,(address_t)c,(uint16_t)d)
#define NOTIFY_TB_CAA32(a,b,c,d)           _notify_tb_caa32((command_t)a,(address_t)b,(address_t)c,(uint32_t)d)
#define NOTIFY_TB_CA88(a,b,c,d)            _notify_tb_ca88((command_t)a,(address_t)b,(uint8_t)c,(uint8_t)d)
#define NOTIFY_TB_CA1616(a,b,c,d)          _notify_tb_ca1616((command_t)a,(address_t)b,(uint16_t)c,(uint16_t)d)
#define NOTIFY_TB_CA3232(a,b,c,d)          _notify_tb_ca3232((command_t)a,(address_t)b,(uint32_t)c,(uint32_t)d)
#define NOTIFY_TB_CA888(a,b,c,d,e)         _notify_tb_ca888((command_t)a,(address_t)b,(uint8_t)c,(uint8_t)d,(uint8_t)e)
#define NOTIFY_TB_CA161616(a,b,c,d,e)      _notify_tb_ca161616((command_t)a,(address_t)b,(uint16_t)c,(uint16_t)d,(uint16_t)e)
#define NOTIFY_TB_CA323232(a,b,c,d,e)      _notify_tb_ca323232((command_t)a,(address_t)b,(uint32_t)c,(uint32_t)d,(uint32_t)e)
#define NOTIFY_TB_CA8816(a,b,c,d,e)        _notify_tb_ca8816((command_t)a,(address_t)b,(uint8_t)c,(uint8_t)d,(uint16_t)e)
#define NOTIFY_TB_CA323216(a,b,c,d,e)      _notify_tb_ca323216((command_t)a,(address_t)b,(uint32_t)c,(uint32_t)d,(uint16_t)e)
#define NOTIFY_TB_CA8888(a,b,c,d,e,f)      _notify_tb_ca8888((command_t)a,(address_t)b,(uint8_t)c,(uint8_t)d,(uint8_t)e,(uint8_t)f)
#define NOTIFY_TB_CA16161616(a,b,c,d,e,f)  _notify_tb_ca16161616((command_t)a,(address_t)b,(uint16_t)c,(uint16_t)d,(uint16_t)e,(uint16_t)f)
#define NOTIFY_TB_CA32323232(a,b,c,d,e,f)  _notify_tb_ca32323232((command_t)a,(address_t)b,(uint32_t)c,(uint32_t)d,(uint32_t)e,(uint32_t)f)
#define NOTIFY_TB_CA88816(a,b,c,d,e,f)     _notify_tb_ca88816((command_t)a,(address_t)b,(uint8_t)c,(uint8_t)d,(uint8_t)e,(uint16_t)f)
#define NOTIFY_TB_CA32323216(a,b,c,d,e,f)  _notify_tb_ca32323216((command_t)a,(address_t)b,(uint32_t)c,(uint32_t)d,(uint32_t)e,(uint16_t)f)
#endif

// Tesbench notification for this SoC (use macro implementation)
#if (MCD_TB_NOTIFY == _USE_MACRO)
#define NOTIFY_TB_CAA(a,b,c)               _NOTIFY_TB_CAA((command_t)a,(address_t)b,(address_t)c)
#define NOTIFY_TB_CA8(a,b,c)               _NOTIFY_TB_CA8((command_t)a,(address_t)b,(uint8_t)c)
#define NOTIFY_TB_CA16(a,b,c)              _NOTIFY_TB_CA16((command_t)a,(address_t)b,(uint16_t)c)
#define NOTIFY_TB_CA32(a,b,c)              _NOTIFY_TB_CA32((command_t)a,(address_t)b,(uint32_t)c)
#define NOTIFY_TB_CAA8(a,b,c,d)            _NOTIFY_TB_CAA8((command_t)a,(address_t)b,(address_t)c,(uint8_t)d)
#define NOTIFY_TB_CAA16(a,b,c,d)           _NOTIFY_TB_CAA16((command_t)a,(address_t)b,(address_t)c,(uint16_t)d)
#define NOTIFY_TB_CAA32(a,b,c,d)           _NOTIFY_TB_CAA32((command_t)a,(address_t)b,(address_t)c,(uint32_t)d)
#define NOTIFY_TB_CA88(a,b,c,d)            _NOTIFY_TB_CA88((command_t)a,(address_t)b,(uint8_t)c,(uint8_t)d)
#define NOTIFY_TB_CA1616(a,b,c,d)          _NOTIFY_TB_CA1616((command_t)a,(address_t)b,(uint16_t)c,(uint16_t)d)
#define NOTIFY_TB_CA3232(a,b,c,d)          _NOTIFY_TB_CA3232((command_t)a,(address_t)b,(uint32_t)c,(uint32_t)d)
#define NOTIFY_TB_CA888(a,b,c,d,e)         _NOTIFY_TB_CA888((command_t)a,(address_t)b,(uint8_t)c,(uint8_t)d,(uint8_t)e)
#define NOTIFY_TB_CA161616(a,b,c,d,e)      _NOTIFY_TB_CA161616((command_t)a,(address_t)b,(uint16_t)c,(uint16_t)d,(uint16_t)e)
#define NOTIFY_TB_CA323232(a,b,c,d,e)      _NOTIFY_TB_CA323232((command_t)a,(address_t)b,(uint32_t)c,(uint32_t)d,(uint32_t)e)
#define NOTIFY_TB_CA8816(a,b,c,d,e)        _NOTIFY_TB_CA8816((command_t)a,(address_t)b,(uint8_t)c,(uint8_t)d,(uint16_t)e)
#define NOTIFY_TB_CA323216(a,b,c,d,e)      _NOTIFY_TB_CA323216((command_t)a,(address_t)b,(uint32_t)c,(uint32_t)d,(uint16_t)e)
#define NOTIFY_TB_CA8888(a,b,c,d,e,f)      _NOTIFY_TB_CA8888((command_t)a,(address_t)b,(uint8_t)c,(uint8_t)d,(uint8_t)e,(uint8_t)f)
#define NOTIFY_TB_CA16161616(a,b,c,d,e,f)  _NOTIFY_TB_CA16161616((command_t)a,(address_t)b,(uint16_t)c,(uint16_t)d,(uint16_t)e,(uint16_t)f)
#define NOTIFY_TB_CA32323232(a,b,c,d,e,f)  _NOTIFY_TB_CA32323232((command_t)a,(address_t)b,(uint32_t)c,(uint32_t)d,(uint32_t)e,(uint32_t)f)
#define NOTIFY_TB_CA88816(a,b,c,d,e,f)     _NOTIFY_TB_CA88816((command_t)a,(address_t)b,(uint8_t)c,(uint8_t)d,(uint8_t)e,(uint16_t)f)
#define NOTIFY_TB_CA32323216(a,b,c,d,e,f)  _NOTIFY_TB_CA32323216((command_t)a,(address_t)b,(uint32_t)c,(uint32_t)d,(uint32_t)e,(uint16_t)f)
#endif

// Tesbench notification for this SoC (disabled)
#if (MCD_TB_NOTIFY == _USE_NONE)
#define NOTIFY_TB_CAA(a,b,c)               {}
#define NOTIFY_TB_CA8(a,b,c)               {}
#define NOTIFY_TB_CA16(a,b,c)              {}
#define NOTIFY_TB_CA32(a,b,c)              {}
#define NOTIFY_TB_CAA8(a,b,c,d)            {}
#define NOTIFY_TB_CAA16(a,b,c,d)           {}
#define NOTIFY_TB_CAA32(a,b,c,d)           {}
#define NOTIFY_TB_CA88(a,b,c,d)            {}
#define NOTIFY_TB_CA1616(a,b,c,d)          {}
#define NOTIFY_TB_CA3232(a,b,c,d)          {}
#define NOTIFY_TB_CA888(a,b,c,d,e)         {}
#define NOTIFY_TB_CA161616(a,b,c,d,e)      {}
#define NOTIFY_TB_CA323232(a,b,c,d,e)      {}
#define NOTIFY_TB_CA8816(a,b,c,d,e)        {}
#define NOTIFY_TB_CA323216(a,b,c,d,e)      {}
#define NOTIFY_TB_CA8888(a,b,c,d,e,f)      {}
#define NOTIFY_TB_CA16161616(a,b,c,d,e,f)  {}
#define NOTIFY_TB_CA32323232(a,b,c,d,e,f)  {}
#define NOTIFY_TB_CA88816(a,b,c,d,e,f)     {}
#define NOTIFY_TB_CA32323216(a,b,c,d,e,f)  {}
#endif


///////////////////////////////////////////////////////////////////////////////
// P A S S   &   F A I L   N O T I F I C A T I O N
///////////////////////////////////////////////////////////////////////////////
#define _SUCCESS()      {}
#define _FAILURE()      { ++error_counter; fail_status=1; }
#define _TERMINATE()    { ++error_counter; fail_status=1; EXIT_PORTAL=0xDEAD; }

// Pass/fail notification for this SoC (use function implementation)
#if (MCD_PF_NOTIFY == _USE_FUNC)
#define SUCCESS()       _success()
#define FAILURE()       _failure()
#define TERMINATE()     _terminate()
#endif

// Pass/fail notification for this SoC (use macro implementation)
#if (MCD_PF_NOTIFY == _USE_MACRO)
#define SUCCESS()       _SUCCESS()
#define FAILURE()       _FAILURE()
#define TERMINATE()     _TERMINATE()
#endif

// Pass/fail notification for this SoC (disabled)
#if (MCD_PF_NOTIFY == _USE_NONE)
#define SUCCESS()       {}
#define FAILURE()       {}
#define TERMINATE()     {}
#endif


///////////////////////////////////////////////////////////////////////////////
// T E S T B E N C H   C O M M U N I C A T I O N
///////////////////////////////////////////////////////////////////////////////
#define _TRIGGER_SET(flag) { \
   CPRINT_ARG0 = (flag_t)(flag); \
   CPRINT_CMD  = (command_t)CPRINT_TRIGGER_SET; \
}
#define _TRIGGER_WAIT(flag) { \
   do { \
      CPRINT_ARG0 = (flag_t)(flag); \
      CPRINT_CMD  = (command_t)CPRINT_TRIGGER_WAIT; \
   } while(!*((flag_status_t*)&(CPRINT_ARG1))); \
}
#define _MB_PUTx(type,mbox,data) { \
   CPRINT_ARG0 = (mailbox_t)    (mbox); \
   CPRINT_ARG1 = (uint_t(type)) (data); \
   CPRINT_CMD  = (command_t)    CPRINT_MB_PUT##type;  \
}
#define _MB_GETx(type,mbox,dst) { \
   do { \
      CPRINT_ARG0 = (mailbox_t) (mbox); \
      CPRINT_CMD  = (command_t) CPRINT_MB_GET##type; \
   } while(!*((mailbox_status_t*) &(CPRINT_ARG1))); \
   *((uint_t(type)*) &(dst)) = *((uint_t(type)*) &(CPRINT_ARG2)); \
}

// Testbench communication for this SoC (use function implementation)
#if (MCD_TB_COM == _USE_FUNC)
#define TRIGGER_SET(x)  _trigger_set((flag_t)x)
#define TRIGGER_WAIT(x) _trigger_wait((flag_t)x)
#define MB_PUT8(x,y)    _mb_put8((mailbox_t)x,(uint8_t)y)
#define MB_PUT16(x,y)   _mb_put16((mailbox_t)x,(uint16_t)y)
#define MB_PUT32(x,y)   _mb_put32((mailbox_t)x,(uint32_t)y)
#define MB_GET8(x,y)    _mb_get8((mailbox_t)(x),(uint8_t*)&(y))
#define MB_GET16(x,y)   _mb_get16((mailbox_t)(x),(uint16_t*)&(y))
#define MB_GET32(x,y)   _mb_get32((mailbox_t)(x),(uint32_t*)&(y))
#endif

// Testbench communication for this SoC (use macro implementation)
#if (MCD_TB_COM == _USE_MACRO)
#define TRIGGER_SET(x)  _TRIGGER_SET(x)
#define TRIGGER_WAIT(x) _TRIGGER_WAIT(x)
#define MB_PUT8(x,y)    _MB_PUTx(8,x,y)
#define MB_PUT16(x,y)   _MB_PUTx(16,x,y)
#define MB_PUT32(x,y)   _MB_PUTx(32,x,y)
#define MB_GET8(x,y)    _MB_GETx(8,x,y)
#define MB_GET16(x,y)   _MB_GETx(16,x,y)
#define MB_GET32(x,y)   _MB_GETx(32,x,y)
#endif

// Testbench communication for this SoC (disabled)
#if (MCD_TB_COM == _USE_NONE)
#define TRIGGER_SET(x)  {}
#define TRIGGER_WAIT(x) {}
#define MB_PUT8(x,y)    {}
#define MB_PUT16(x,y)   {}
#define MB_PUT32(x,y)   {}
#define MB_GET8(x,y)    {}
#define MB_GET16(x,y)   {}
#define MB_GET32(x,y)   {}
#endif


///////////////////////////////////////////////////////////////////////////////
// C - A P I  D E F I N I T I O N
///////////////////////////////////////////////////////////////////////////////

// C API Definition (no pass/fail notification)
#define _DEBUG(source, comment) { \
   NOTIFY_TB_CAA(CPRINT_DEBUG, (address_t)(source), (address_t)(comment)) }
        
#define _TRACE(source, comment) { \
   NOTIFY_TB_CAA(CPRINT_TRACE, (address_t)(source), (address_t)(comment)) }
        
#define _INFO(source, comment) { \
   NOTIFY_TB_CAA(CPRINT_INFO, (address_t)(source), (address_t)(comment)) }
        
#define _WARN(source, comment) { \
   NOTIFY_TB_CAA(CPRINT_WARN, (address_t)(source), (address_t)(comment)); \
   FAILURE(); }
  
#define _ERROR(source, comment) { \
   NOTIFY_TB_CAA(CPRINT_ERROR, (address_t)(source), (address_t)(comment)); \
   FAILURE(); }
  
#define _FATAL(source, comment) { \
   NOTIFY_TB_CAA(CPRINT_FATAL, (address_t)(source), (address_t)(comment)); \
   TERMINATE(); }

#define _SHOW8(source, comment, value) { \
   NOTIFY_TB_CAA8(CPRINT_SHOW8, (address_t)(source), (address_t)(comment), (uint8_t)(value)) }

#define _SHOW16(source, comment, value) { \
   NOTIFY_TB_CAA16(CPRINT_SHOW16, (address_t)(source), (address_t)(comment), (uint16_t)(value)) }

#define _SHOW32(source, comment, value) { \
   NOTIFY_TB_CAA32(CPRINT_SHOW32, (address_t)(source), (address_t)(comment), (uint32_t)(value)) }


///////////////////////////////////////////////////////////////////////////////
// C API Macro Definitions (all "x" macros notify TB on failures only)
///////////////////////////////////////////////////////////////////////////////
#define _Rx(x, addr, dst) { *((uint_t(x)*)&(dst)) = *((vuint_t(x)*)&(addr)); }

#define _Wx(x, addr, data) { *((vuint_t(x)*)&(addr)) = (uint_t(x))(data); }

#define _BSETx(x, addr, msk) { *((vuint_t(x)*)&(addr)) |= (uint_t(x))(msk); }

#define _BCLRx(x, addr, msk) { *((vuint_t(x)*)&(addr)) &= ~(uint_t(x))(msk); }

#define _BINVx(x, addr, msk) { *((vuint_t(x)*)&(addr)) ^= (uint_t(x))(msk); }

#define _REx(x, addr, exp) { \
   uint_t(x) actual_data = *((vuint_t(x)*)&(addr)); \
   if (actual_data != (uint_t(x))(exp)) { \
      NOTIFY_TB_CA##x##x(CPRINT_E_RE##x, (address_t)&(addr), actual_data, (uint_t(x))(exp)); \
      FAILURE(); \
   } \
}

#define _REMx(x, addr, exp, msk) { \
   uint_t(x) actual_data = *((vuint_t(x)*)&(addr)); \
   if ((actual_data & (uint_t(x))(msk)) != ((uint_t(x))(exp) & (uint_t(x))(msk))) { \
      NOTIFY_TB_CA##x##x##x(CPRINT_E_REM##x, (address_t)&(addr), actual_data, (uint_t(x))(exp), (uint_t(x))(msk)); \
      FAILURE(); \
   } \
}

#define _RREx(x, addr, dst, exp) { \
   *((uint_t(x)*)&(dst)) = *((vuint_t(x)*)&(addr)); \
   if (*((uint_t(x)*)&(dst)) != (uint_t(x))(exp)) { \
      NOTIFY_TB_CA##x##x(CPRINT_E_RRE##x, (address_t)&(addr), *((uint_t(x)*)&(dst)), (uint_t(x))(exp)); \
      FAILURE(); \
   } \
}

#define _RREMx(x, addr, dst, exp, msk) { \
   *((uint_t(x)*)&(dst)) = *((vuint_t(x)*)&(addr)); \
   if ((*((uint_t(x)*)&(dst)) & (uint_t(x))(msk)) != ((uint_t(x))(exp) & (uint_t(x))(msk))) { \
      NOTIFY_TB_CA##x##x##x(CPRINT_E_RREM##x, (address_t)&(addr), *((uint_t(x)*)&(dst)), (uint_t(x))(exp), (uint_t(x))(msk)); \
      FAILURE(); \
   } \
}

#define _WRx(x, addr, data) { \
   uint_t(x) actual_data; \
   *((vuint_t(x)*)&(addr)) = (uint_t(x))(data); \
   actual_data = *((vuint_t(x)*)&(addr)); \
   if (actual_data != (uint_t(x))(data)) { \
      NOTIFY_TB_CA##x##x(CPRINT_E_WR##x, (address_t)&(addr), (uint_t(x))(data), actual_data); \
      FAILURE(); \
   } \
}

#define _WRMx(x, addr, data, msk) { \
   uint_t(x) actual_data; \
   *((vuint_t(x)*)&(addr)) = (uint_t(x))(data); \
   actual_data = *((vuint_t(x)*)&(addr)); \
   if ((actual_data & (uint_t(x))(msk)) != ((uint_t(x))(data) & (uint_t(x))(msk))) { \
      NOTIFY_TB_CA##x##x##x(CPRINT_E_WRM##x, (address_t)&(addr), (uint_t(x))(data), actual_data, (uint_t(x))(msk)); \
      FAILURE(); \
   } \
}

#define _WREx(x, addr, data, exp) { \
   uint_t(x) actual_data; \
   *((vuint_t(x)*)&(addr)) = (uint_t(x))(data); \
   actual_data = *((vuint_t(x)*)&(addr)); \
   if (actual_data != (uint_t(x))(exp)) { \
      NOTIFY_TB_CA##x##x##x(CPRINT_E_WRE##x, (address_t)&(addr), (uint_t(x))(data), actual_data, (uint_t(x))(exp)); \
      FAILURE(); \
   } \
}

#define _WREMx(x, addr, data, exp, msk) { \
   uint_t(x) actual_data; \
   *((vuint_t(x)*)&(addr)) = (uint_t(x))(data); \
   actual_data = *((vuint_t(x)*)&(addr)); \
   if ((actual_data & (uint_t(x))(msk)) != ((uint_t(x))(exp) & (uint_t(x))(msk))) { \
     NOTIFY_TB_CA##x##x##x##x(CPRINT_E_WREM##x, (address_t)&(addr), (uint_t(x))(data), actual_data, (uint_t(x))(exp), (uint_t(x))(msk)); \
      FAILURE(); \
   } \
}

#define _WAIT_REx(x, address, expect_data, read_attempts) { \
   uint_t(x) actual_data; \
   if (!(uint16_t)(read_attempts)) { \
      do { \
         actual_data = *((vuint_t(x)*)&(address)); \
      } while (actual_data != (uint_t(x))(expect_data)); \
      NOTIFY_TB_CA##x##x##16(CPRINT_S_WAIT_RE##x, (address_t)&(address), actual_data, (uint_t(x))(expect_data), (uint16_t)(read_attempts)); \
      SUCCESS(); \
   } else { \
      uint16_t count=0; \
      actual_data = *((vuint_t(x)*)&(address)); \
      while ((actual_data != (uint_t(x))(expect_data)) && (count < (uint16_t)(read_attempts))) { \
         actual_data = *((vuint_t(x)*)&(address)); \
         ++count; \
      } \
      if (count < (uint16_t)(read_attempts)) { \
         NOTIFY_TB_CA##x##x##16(CPRINT_S_WAIT_RE8, (address_t)&(address), actual_data, (uint_t(x))(expect_data), (uint16_t)(read_attempts)); \
         SUCCESS(); \
      } else { \
         NOTIFY_TB_CA##x##x##16(CPRINT_E_WAIT_RE8, (address_t)&(address), actual_data, (uint_t(x))(expect_data), (uint16_t)(read_attempts)); \
         FAILURE(); \
      } \
   } \
}

#define _WAIT_REMx(x, address, expect_data, mask, read_attempts) { \
   uint_t(x) actual_data; \
   if (!(uint16_t)(read_attempts)) { \
      do { \
         actual_data = *((vuint_t(x)*)&(address)); \
      } while ((actual_data & (uint_t(x))(mask)) != ((uint_t(x))(expect_data) & (uint_t(x))(mask))); \
      NOTIFY_TB_CA##x##x##x##16(CPRINT_S_WAIT_REM##x, (address_t)&(address), actual_data, (uint_t(x))(expect_data), (uint_t(x))(mask), (uint16_t)(read_attempts)); \
      SUCCESS(); \
   } else { \
      uint16_t count=0; \
      actual_data = *((vuint_t(x)*)&(address)); \
      while (((actual_data & (uint_t(x))(mask)) != ((uint_t(x))(expect_data) & (uint_t(x))(mask))) && (count < (uint16_t)(read_attempts))) { \
         actual_data = *((vuint_t(x)*)&(address)); \
         ++count; \
      } \
      if (count < (uint16_t)(read_attempts)) { \
         NOTIFY_TB_CA##x##x##x##16(CPRINT_S_WAIT_REM##x, (address_t)&(address), actual_data, (uint_t(x))(expect_data), (uint_t(x))(mask), (uint16_t)(read_attempts)); \
         SUCCESS(); \
      } else { \
         NOTIFY_TB_CA##x##x##x##16(CPRINT_E_WAIT_REM##x, (address_t)&(address), actual_data, (uint_t(x))(expect_data), (uint_t(x))(mask), (uint16_t)(read_attempts)); \
         FAILURE(); \
      } \
   } \
}

// Define macro wrappers for later use
#if (_NOTIFY_FAIL==1 && _NOTIFY_PASS==0)
#define _R8(addr,dst)               _Rx(8,addr,dst);
#define _R16(addr,dst)              _Rx(16,addr,dst);
#define _R32(addr,dst)              _Rx(32,addr,dst);
#define _W8(addr,data)              _Wx(8,addr,data);
#define _W16(addr,data)             _Wx(16,addr,data);
#define _W32(addr,data)             _Wx(32,addr,data);
#define _BCLR8(addr,msk)            _BCLRx(8,addr,msk)
#define _BCLR16(addr,msk)           _BCLRx(16,addr,msk)
#define _BCLR32(addr,msk)           _BCLRx(32,addr,msk)
#define _BINV8(addr,msk)            _BINVx(8,addr,msk)
#define _BINV16(addr,msk)           _BINVx(16,addr,msk)
#define _BINV32(addr,msk)           _BINVx(32,addr,msk)
#define _BSET8(addr,msk)            _BSETx(8,addr,msk)
#define _BSET16(addr,msk)           _BSETx(16,addr,msk)
#define _BSET32(addr,msk)           _BSETx(32,addr,msk)
#define _RE8(addr,exp)              _REx(8,addr,exp)
#define _RE16(addr,exp)             _REx(16,addr,exp)
#define _RE32(addr,exp)             _REx(32,addr,exp)
#define _REM8(addr,exp,msk)         _REMx(8,addr,exp,msk)
#define _REM16(addr,exp,msk)        _REMx(16,addr,exp,msk)
#define _REM32(addr,exp,msk)        _REMx(32,addr,exp,msk)
#define _RRE8(addr,dst,exp)         _RREx(8,addr,dst,exp)
#define _RRE16(addr,dst,exp)        _RREx(16,addr,dst,exp)
#define _RRE32(addr,dst,exp)        _RREx(32,addr,dst,exp)
#define _RREM8(addr,dst,exp,msk)    _RREMx(8,addr,dst,exp,msk)
#define _RREM16(addr,dst,exp,msk)   _RREMx(16,addr,dst,exp,msk)
#define _RREM32(addr,dst,exp,msk)   _RREMx(32,addr,dst,exp,msk)
#define _WR8(addr,data)             _WRx(8,addr,data)
#define _WR16(addr,data)            _WRx(16,addr,data)
#define _WR32(addr,data)            _WRx(32,addr,data)
#define _WRM8(addr,data,msk)        _WRMx(8,addr,data,msk)
#define _WRM16(addr,data,msk)       _WRMx(16,addr,data,msk)
#define _WRM32(addr,data,msk)       _WRMx(32,addr,data,msk)
#define _WRE8(addr,data,exp)        _WREx(8,addr,data,exp)
#define _WRE16(addr,data,exp)       _WREx(16,addr,data,exp)
#define _WRE32(addr,data,exp)       _WREx(32,addr,data,exp)
#define _WREM8(addr,data,exp,msk)   _WREMx(8,addr,data,exp,msk)
#define _WREM16(addr,data,exp,msk)  _WREMx(16,addr,data,exp,msk)
#define _WREM32(addr,data,exp,msk)  _WREMx(32,addr,data,exp,msk)
#define _WAIT_RE8(addr,exp,i)       _WAIT_REx(8,addr,exp,i);
#define _WAIT_RE16(addr,exp,i)      _WAIT_REx(16,addr,exp,i);
#define _WAIT_RE32(addr,exp,i)      _WAIT_REx(32,addr,exp,i);
#define _WAIT_REM8(addr,exp,msk,i)  _WAIT_REMx(8,addr,exp,msk,i);
#define _WAIT_REM16(addr,exp,msk,i) _WAIT_REMx(16,addr,exp,msk,i);
#define _WAIT_REM32(addr,exp,msk,i) _WAIT_REMx(32,addr,exp,msk,i);
#endif  // (_NOTIFY_FAIL==1 && _NOTIFY_PASS==0)


///////////////////////////////////////////////////////////////////////////////
// C API Macro Definitions (all "y" macros notify TB on success and failures)
///////////////////////////////////////////////////////////////////////////////
#define _Ry(y, addr, dst) { \
   *((uint_t(y)*)&(dst)) = *((vuint_t(y)*)&(addr)); \
   NOTIFY_TB_CA##y(CPRINT_R##y, (address_t)&(addr), *((uint_t(y)*)&(dst))); \
   SUCCESS(); \
}

#define _Wy(y, addr, data) { \
   *((vuint_t(y)*)&(addr)) = (uint_t(y))(data); \
   NOTIFY_TB_CA##y(CPRINT_W##y, (address_t)&(addr), (uint_t(y))(data)); \
   SUCCESS(); \
}

#define _BSETy(y, addr, msk) { \
   uint_t(y) initial_data, final_data; \
   initial_data = *((vuint_t(y)*)&(addr)); \
   *((vuint_t(y)*)&(addr)) |= (uint_t(y))(msk); \
   final_data = *((vuint_t(y)*)&(addr)); \
   NOTIFY_TB_CA##y##y##y(CPRINT_BSET##y, (address_t)&(addr), (uint_t(y))(initial_data), (uint_t(y))(msk), (uint_t(y))(final_data)); \
   SUCCESS(); \
}

#define _BCLRy(y, addr, msk) { \
   uint_t(y) initial_data, final_data; \
   initial_data = *((vuint_t(y)*)&(addr)); \
   *((vuint_t(y)*)&(addr)) &= ~(uint_t(y))(msk); \
   final_data = *((vuint_t(y)*)&(addr)); \
   NOTIFY_TB_CA##y##y##y(CPRINT_BCLR##y, (address_t)&(addr), (uint_t(y))(initial_data), (uint_t(y))(msk), (uint_t(y))(final_data)); \
   SUCCESS(); \
}

#define _BINVy(y, addr, msk) { \
   uint_t(y) initial_data, final_data; \
   initial_data = *((vuint_t(y)*)&(addr)); \
   *((vuint_t(y)*)&(addr)) ^= (uint_t(y))(msk); \
   final_data = *((vuint_t(y)*)&(addr)); \
   NOTIFY_TB_CA##y##y##y(CPRINT_BINV##y, (address_t)&(addr), (uint_t(y))(initial_data), (uint_t(y))(msk), (uint_t(y))(final_data)); \
   SUCCESS(); \
}

#define _REy(y, addr, exp) {\
   uint_t(y) actual_data = *((vuint_t(y)*)&(addr)); \
   if (actual_data == (uint_t(y))(exp)) { \
      NOTIFY_TB_CA##y##y(CPRINT_S_RE##y, (address_t)&(addr), actual_data, (uint_t(y))(exp)); \
      SUCCESS(); \
   } else { \
      NOTIFY_TB_CA##y##y(CPRINT_E_RE##y, (address_t)&(addr), actual_data, (uint_t(y))(exp)); \
      FAILURE(); \
   } \
}

#define _REMy(y, addr, exp, msk) { \
   uint_t(y) actual_data = *((vuint_t(y)*)&(addr)); \
   if ((actual_data & (uint_t(y))(msk)) == ((uint_t(y))(exp) & (uint_t(y))(msk))) { \
      NOTIFY_TB_CA##y##y##y(CPRINT_S_REM##y, (address_t)&(addr), actual_data, (uint_t(y))(exp), (uint_t(y))(msk)); \
     SUCCESS(); \
   } else { \
      NOTIFY_TB_CA##y##y##y(CPRINT_E_REM##y, (address_t)&(addr), actual_data, (uint_t(y))(exp), (uint_t(y))(msk)); \
      FAILURE(); \
   } \
}

#define _RREy(y, addr, dst, exp) { \
   *((uint_t(y)*)&(dst)) = *((vuint_t(y)*)&(addr)); \
   if (*((uint_t(y)*)&(dst)) == (uint_t(y))(exp)) { \
      NOTIFY_TB_CA##y##y(CPRINT_S_RRE##y, (address_t)&(addr), *((uint_t(y)*)&(dst)), (uint8_t)(exp)); \
      SUCCESS(); \
   } else { \
      NOTIFY_TB_CA##y##y(CPRINT_E_RRE##y, (address_t)&(addr), *((uint_t(y)*)&(dst)), (uint8_t)(exp)); \
      FAILURE(); \
   } \
}

#define _RREMy(y, addr, dst, exp, msk) { \
   *((uint_t(y)*)&(dst)) = *((vuint_t(y)*)&(addr)); \
   if ((*((uint_t(y)*)&(dst)) & (uint_t(y))(msk)) == ((uint_t(y))(exp) & (uint_t(y))(msk))) { \
      NOTIFY_TB_CA##y##y##y(CPRINT_S_RREM##y, (address_t)&(addr), *((uint_t(y)*)&(dst)), (uint_t(y))(exp), (uint_t(y))(msk)); \
      SUCCESS(); \
   } else { \
      NOTIFY_TB_CA##y##y##y(CPRINT_E_RREM##y, (address_t)&(addr), *((uint_t(y)*)&(dst)), (uint_t(y))(exp), (uint_t(y))(msk)); \
      FAILURE(); \
   } \
}

#define _WRy(y, addr, data) { \
   uint_t(y) actual_data; \
   *((vuint_t(y)*)&(addr)) = (uint_t(y))(data); \
   actual_data = *((vuint_t(y)*)&(addr)); \
   if (actual_data == (uint_t(y))(data)) { \
      NOTIFY_TB_CA##y##y(CPRINT_S_WR##y, (address_t)&(addr), (uint_t(y))(data), actual_data); \
      SUCCESS(); \
   } else { \
      NOTIFY_TB_CA##y##y(CPRINT_E_WR##y, (address_t)&(addr), (uint_t(y))(data), actual_data); \
      FAILURE(); \
   } \
}

#define _WRMy(y, addr, data, msk) { \
   uint_t(y) actual_data; \
   *((vuint_t(y)*)&(addr)) = (uint_t(y))(data); \
   actual_data = *((vuint_t(y)*)&(addr)); \
   if ((actual_data & (uint_t(y))(msk)) == ((uint_t(y))(data) & (uint_t(y))(msk))) { \
      NOTIFY_TB_CA##y##y##y(CPRINT_S_WRM##y, (address_t)&(addr), (uint_t(y))(data), actual_data, (uint_t(y))(msk)); \
      SUCCESS(); \
   } else { \
      NOTIFY_TB_CA##y##y##y(CPRINT_E_WRM##y, (address_t)&(addr), (uint_t(y))(data), actual_data, (uint_t(y))(msk)); \
      FAILURE(); \
   } \
}

#define _WREy(y, addr, data, exp) { \
   uint_t(y) actual_data; \
   *((vuint_t(y)*)&(addr)) = (uint_t(y))(data); \
   actual_data = *((vuint_t(y)*)&(addr)); \
   if (actual_data == (uint_t(y))(exp)) { \
      NOTIFY_TB_CA##y##y##y(CPRINT_S_WRE##y, (address_t)&(addr), (uint_t(y))(data), actual_data, (uint_t(y))(exp)); \
      SUCCESS(); \
   } else { \
      NOTIFY_TB_CA##y##y##y(CPRINT_E_WRE##y, (address_t)&(addr), (uint_t(y))(data), actual_data, (uint_t(y))(exp)); \
      FAILURE(); \
   } \
}

#define _WREMy(y, addr, data, exp, msk) { \
   uint_t(y) actual_data; \
   *((vuint_t(y)*)&(addr)) = (uint_t(y))(data); \
   actual_data = *((vuint_t(y)*)&(addr)); \
   if ((actual_data & (uint_t(y))(msk)) == ((uint_t(y))(exp) & (uint_t(y))(msk))) { \
      NOTIFY_TB_CA##y##y##y##y(CPRINT_S_WREM##y, (address_t)&(addr), (uint_t(y))(data), actual_data, (uint_t(y))(exp), (uint_t(y))(msk)); \
      SUCCESS(); \
   } else { \
      NOTIFY_TB_CA##y##y##y##y(CPRINT_E_WREM##y, (address_t)&(addr), (uint_t(y))(data), actual_data, (uint_t(y))(exp), (uint_t(y))(msk)); \
      FAILURE(); \
   } \
}

#define _WAIT_REy(y, address, expect_data, read_attempts) { \
   uint_t(y) actual_data; \
   if (!(uint16_t)(read_attempts)) { \
      do { \
         actual_data = *((vuint_t(y)*)&(address)); \
      } while (actual_data != (uint_t(y))(expect_data)); \
      NOTIFY_TB_CA##y##y##16(CPRINT_S_WAIT_RE##y, (address_t)&(address), actual_data, (uint_t(y))(expect_data), (uint16_t)(read_attempts)); \
      SUCCESS(); \
   } else { \
      uint16_t count=0; \
      actual_data = *((vuint_t(y)*)&(address)); \
      while ((actual_data != (uint_t(y))(expect_data)) && (count < (uint16_t)(read_attempts))) { \
         actual_data = *((vuint_t(y)*)&(address)); \
         ++count; \
      } \
      if (count < (uint16_t)(read_attempts)) { \
         NOTIFY_TB_CA##y##y##16(CPRINT_S_WAIT_RE##y, (address_t)&(address), actual_data, (uint_t(y))(expect_data), (uint16_t)(read_attempts)); \
         SUCCESS(); \
      } else { \
         NOTIFY_TB_CA##y##y##16(CPRINT_E_WAIT_RE##y, (address_t)&(address), actual_data, (uint_t(y))(expect_data), (uint16_t)(read_attempts)); \
         FAILURE(); \
      } \
   } \
}

#define _WAIT_REMy(y, address, expect_data, mask, read_attempts) { \
   uint_t(y) actual_data; \
   if (!(uint16_t)(read_attempts)) { \
      do { \
         actual_data = *((vuint_t(y)*)&(address)); \
      } while ((actual_data & (uint_t(y))(mask)) != ((uint_t(y))(expect_data) & (uint_t(y))(mask))); \
      NOTIFY_TB_CA##y##y##y##16(CPRINT_S_WAIT_REM##y, (address_t)&(address), actual_data, (uint_t(y))(expect_data), (uint_t(y))(mask), (uint16_t)(read_attempts)); \
      SUCCESS(); \
   } else { \
      uint16_t count=0; \
      actual_data = *((vuint_t(y)*)&(address)); \
      while (((actual_data & (uint_t(y))(mask)) != ((uint_t(y))(expect_data) & (uint_t(y))(mask))) && (count < (uint16_t)(read_attempts))) { \
         actual_data = *((vuint_t(y)*)&(address)); \
         ++count; \
      } \
      if (count < (uint16_t)(read_attempts)) { \
         NOTIFY_TB_CA##y##y##y##16(CPRINT_S_WAIT_REM##y, (address_t)&(address), actual_data, (uint_t(y))(expect_data), (uint_t(y))(mask), (uint16_t)(read_attempts)); \
         SUCCESS(); \
      } else { \
         NOTIFY_TB_CA##y##y##y##16(CPRINT_E_WAIT_REM##y, (address_t)&(address), actual_data, (uint_t(y))(expect_data), (uint_t(y))(mask), (uint16_t)(read_attempts)); \
         FAILURE(); \
      } \
   } \
}

// Define macro wrappers for later use
#if (_NOTIFY_FAIL==1 && _NOTIFY_PASS==1)
#define _R8(addr,dst)               _Ry(8,addr,dst);
#define _R16(addr,dst)              _Ry(16,addr,dst);
#define _R32(addr,dst)              _Ry(32,addr,dst);
#define _W8(addr,data)              _Wy(8,addr,data);
#define _W16(addr,data)             _Wy(16,addr,data);
#define _W32(addr,data)             _Wy(32,addr,data);
#define _BCLR8(addr,msk)            _BCLRy(8,addr,msk)
#define _BCLR16(addr,msk)           _BCLRy(16,addr,msk)
#define _BCLR32(addr,msk)           _BCLRy(32,addr,msk)
#define _BINV8(addr,msk)            _BINVy(8,addr,msk)
#define _BINV16(addr,msk)           _BINVy(16,addr,msk)
#define _BINV32(addr,msk)           _BINVy(32,addr,msk)
#define _BSET8(addr,msk)            _BSETy(8,addr,msk)
#define _BSET16(addr,msk)           _BSETy(16,addr,msk)
#define _BSET32(addr,msk)           _BSETy(32,addr,msk)
#define _RE8(addr,exp)              _REy(8,addr,exp)
#define _RE16(addr,exp)             _REy(16,addr,exp)
#define _RE32(addr,exp)             _REy(32,addr,exp)
#define _REM8(addr,exp,msk)         _REMy(8,addr,exp,msk)
#define _REM16(addr,exp,msk)        _REMy(16,addr,exp,msk)
#define _REM32(addr,exp,msk)        _REMy(32,addr,exp,msk)
#define _RRE8(addr,dst,exp)         _RREy(8,addr,dst,exp)
#define _RRE16(addr,dst,exp)        _RREy(16,addr,dst,exp)
#define _RRE32(addr,dst,exp)        _RREy(32,addr,dst,exp)
#define _RREM8(addr,dst,exp,msk)    _RREMy(8,addr,dst,exp,msk)
#define _RREM16(addr,dst,exp,msk)   _RREMy(16,addr,dst,exp,msk)
#define _RREM32(addr,dst,exp,msk)   _RREMy(32,addr,dst,exp,msk)
#define _WR8(addr,data)             _WRy(8,addr,data)
#define _WR16(addr,data)            _WRy(16,addr,data)
#define _WR32(addr,data)            _WRy(32,addr,data)
#define _WRM8(addr,data,msk)        _WRMy(8,addr,data,msk)
#define _WRM16(addr,data,msk)       _WRMy(16,addr,data,msk)
#define _WRM32(addr,data,msk)       _WRMy(32,addr,data,msk)
#define _WRE8(addr,data,exp)        _WREy(8,addr,data,exp)
#define _WRE16(addr,data,exp)       _WREy(16,addr,data,exp)
#define _WRE32(addr,data,exp)       _WREy(32,addr,data,exp)
#define _WREM8(addr,data,exp,msk)   _WREMy(8,addr,data,exp,msk)
#define _WREM16(addr,data,exp,msk)  _WREMy(16,addr,data,exp,msk)
#define _WREM32(addr,data,exp,msk)  _WREMy(32,addr,data,exp,msk)
#define _WAIT_RE8(addr,exp,i)       _WAIT_REy(8,addr,exp,i);
#define _WAIT_RE16(addr,exp,i)      _WAIT_REy(16,addr,exp,i);
#define _WAIT_RE32(addr,exp,i)      _WAIT_REy(32,addr,exp,i);
#define _WAIT_REM8(addr,exp,msk,i)  _WAIT_REMy(8,addr,exp,msk,i);
#define _WAIT_REM16(addr,exp,msk,i) _WAIT_REMy(16,addr,exp,msk,i);
#define _WAIT_REM32(addr,exp,msk,i) _WAIT_REMy(32,addr,exp,msk,i);
#endif  // (_NOTIFY_FAIL==1 && _NOTIFY_PASS==1)


// C API Definition (use funcion implementation)
#if (MCD_C_API == _USE_FUNC)
#define R8(addr,dst)                    _r8((vuint8_t*)&(addr), (uint8_t*)&(dst))
#define R16(addr,dst)                   _r16((vuint16_t*)&(addr), (uint16_t*)&(dst))
#define R32(addr,dst)                   _r32((vuint32_t*)&(addr), (uint32_t*)&(dst))
#define RE8(addr,exp)                   _re8((vuint8_t*)&(addr), (uint8_t)(exp))
#define RE16(addr,exp)                  _re16((vuint16_t*)&(addr), (uint16_t)(exp))
#define RE32(addr,exp)                  _re32((vuint32_t*)&(addr), (uint32_t)(exp))
#define REM8(addr,exp,msk)              _rem8((vuint8_t*)&(addr), (uint8_t)(exp), (uint8_t)(msk))
#define REM16(addr,exp,msk)             _rem16((vuint16_t*)&(addr), (uint16_t)(exp), (uint16_t)(msk))
#define REM32(addr,exp,msk)             _rem32((vuint32_t*)&(addr), (uint32_t)(exp), (uint32_t)(msk))
#define RRE8(addr,dst,exp)              _rre8((vuint8_t*)&(addr), (uint8_t*)&(dst), (uint8_t)(exp))
#define RRE16(addr,dst,exp)             _rre16((vuint16_t*)&(addr), (uint16_t*)&(dst), (uint16_t)(exp))
#define RRE32(addr,dst,exp)             _rre32((vuint32_t*)&(addr), (uint32_t*)&(dst), (uint32_t)(exp))
#define RREM8(addr,dst,exp,msk)         _rrem8((vuint8_t*)&(addr), (uint8_t*)&(dst), (uint8_t)(exp), (uint8_t)(msk))
#define RREM16(addr,dst,exp,msk)        _rrem16((vuint16_t*)&(addr), (uint16_t*)&(dst), (uint16_t)(exp), (uint16_t)(msk))
#define RREM32(addr,dst,exp,msk)        _rrem32((vuint32_t*)&(addr), (uint32_t*)&(dst), (uint32_t)(exp), (uint32_t)(msk))
#define W8(addr,data)                   _w8((vuint8_t*)&(addr), (uint8_t)(data))
#define W16(addr,data)                  _w16((vuint16_t*)&(addr), (uint16_t)(data))
#define W32(addr,data)                  _w32((vuint32_t*)&(addr), (uint32_t)(data))
#define WR8(addr,data)                  _wr8((vuint8_t*)&(addr), (uint8_t)(data))
#define WR16(addr,data)                 _wr16((vuint16_t*)&(addr), (uint16_t)(data))
#define WR32(addr,data)                 _wr32((vuint32_t*)&(addr), (uint32_t)(data))
#define WRM8(addr,data,msk)             _wrm8((vuint8_t*)&(addr), (uint8_t)(data), (uint8_t)(msk))
#define WRM16(addr,data,msk)            _wrm16((vuint16_t*)&(addr), (uint16_t)(data), (uint16_t)(msk))
#define WRM32(addr,data,msk)            _wrm32((vuint32_t*)&(addr), (uint32_t)(data), (uint32_t)(msk))
#define WRE8(addr,data,exp)             _wre8((vuint8_t*)&(addr), (uint8_t)(data), (uint8_t)(exp))
#define WRE16(addr,data,exp)            _wre16((vuint16_t*)&(addr), (uint16_t)(data), (uint16_t)(exp))
#define WRE32(addr,data,exp)            _wre32((vuint32_t*)&(addr), (uint32_t)(data), (uint32_t)(exp))
#define WREM8(addr,data,exp,msk)        _wrem8((vuint8_t*)&(addr), (uint8_t)(data), (uint8_t)(exp), (uint8_t)(msk))
#define WREM16(addr,data,exp,msk)       _wrem16((vuint16_t*)&(addr), (uint16_t)(data), (uint16_t)(exp), (uint16_t)(msk))
#define WREM32(addr,data,exp,msk)       _wrem32((vuint32_t*)&(addr), (uint32_t)(data), (uint32_t)(exp), (uint32_t)(msk))
#define WAIT_RE8(addr,exp,i)            _wait_re8((vuint8_t*)&(addr), (uint8_t)(exp), (uint16_t)(i))
#define WAIT_RE16(addr,exp,i)           _wait_re16((vuint16_t*)&(addr), (uint16_t)(exp), (uint16_t)(i))
#define WAIT_RE32(addr,exp,i)           _wait_re32((vuint32_t*)&(addr), (uint32_t)(exp), (uint16_t)(i))
#define WAIT_REM8(addr,exp,msk,i)       _wait_rem8((vuint8_t*)&(addr), (uint8_t)(exp), (uint8_t)(msk), (uint16_t)(i))
#define WAIT_REM16(addr,exp,msk,i)      _wait_rem16((vuint16_t*)&(addr), (uint16_t)(exp), (uint16_t)(msk), (uint16_t)(i))
#define WAIT_REM32(addr,exp,msk,i)      _wait_rem32((vuint32_t*)&(addr), (uint32_t)(exp), (uint32_t)(msk), (uint16_t)(i))
#define BSET8(addr,msk)                 _bset8((vuint8_t*)&(addr), (uint8_t)(msk))
#define BSET16(addr,msk)                _bset16((vuint16_t*)&(addr), (uint16_t)(msk))
#define BSET32(addr,msk)                _bset32((vuint32_t*)&(addr), (uint32_t)(msk))
#define BCLR8(addr,msk)                 _bclr8((vuint8_t*)&(addr), (uint8_t)(msk))
#define BCLR16(addr,msk)                _bclr16((vuint16_t*)&(addr), (uint16_t)(msk))
#define BCLR32(addr,msk)                _bclr32((vuint32_t*)&(addr), (uint32_t)(msk))
#define BINV8(addr,msk)                 _binv8((vuint8_t*)&(addr), (uint8_t)(msk))
#define BINV16(addr,msk)                _binv16((vuint16_t*)&(addr), (uint16_t)(msk))
#define BINV32(addr,msk)                _binv32((vuint32_t*)&(addr), (uint32_t)(msk))
#define DEBUG(src,cmt)                  _debug((char*)src, (char*)cmt)
#define TRACE(src,cmt)                  _trace((char*)src, (char*)cmt)
#define INFO(src,cmt)                   _info((char*)src, (char*)cmt)
#define WARN(src,cmt)                   _warn((char*)src, (char*)cmt)
#define ERROR(src,cmt)                  _error((char*)src, (char*)cmt)
#define FATAL(src,cmt)                  _fatal((char*)src, (char*)cmt)
#define SHOW8(src,cmt,val)              _show8((char*)src, (char*)cmt, (uint8_t)val)
#define SHOW16(src,cmt,val)             _show16((char*)src, (char*)cmt, (uint16_t)val)
#define SHOW32(src,cmt,val)             _show32((char*)src,(char*)cmt, (uint32_t)val)
#endif  // (MCD_C_API == _USE_FUNC)


#if (MCD_C_API == _USE_MACRO)
#define R8(addr,dst)                    _R8(addr,dst)
#define R16(addr,dst)                   _R16(addr,dst)
#define R32(addr,dst)                   _R32(addr,dst)
#define RE8(addr,exp)                   _RE8(addr,exp)
#define RE16(addr,exp)                  _RE16(addr,exp)
#define RE32(addr,exp)                  _RE32(addr,exp)
#define REM8(addr,exp,msk)              _REM8(addr,exp,msk)
#define REM16(addr,exp,msk)             _REM16(addr,exp,msk)
#define REM32(addr,exp,msk)             _REM32(addr,exp,msk)
#define RRE8(addr,dst,exp)              _RRE8(addr,dst,exp)
#define RRE16(addr,dst,exp)             _RRE16(addr,dst,exp)
#define RRE32(addr,dst,exp)             _RRE32(addr,dst,exp)
#define RREM8(addr,dst,exp,msk)         _RREM8(addr,dst,exp,msk)
#define RREM16(addr,dst,exp,msk)        _RREM16(addr,dst,exp,msk)
#define RREM32(addr,dst,exp,msk)        _RREM32(addr,dst,exp,msk)
#define W8(addr,data)                   _W8(addr,data)
#define W16(addr,data)                  _W16(addr,data)
#define W32(addr,data)                  _W32(addr,data)
#define WR8(addr,data)                  _WR8(addr,data)
#define WR16(addr,data)                 _WR16(addr,data)
#define WR32(addr,data)                 _WR32(addr,data)
#define WRM8(addr,data,msk)             _WRM8(addr,data,msk)
#define WRM16(addr,data,msk)            _WRM16(addr,data,msk)
#define WRM32(addr,data,msk)            _WRM32(addr,data,msk)
#define WRE8(addr,data,exp)             _WRE8(addr,data,exp)
#define WRE16(addr,data,exp)            _WRE16(addr,data,exp)
#define WRE32(addr,data,exp)            _WRE32(addr,data,exp)
#define WREM8(addr,data,exp,msk)        _WREM8(addr,data,exp,msk)
#define WREM16(addr,data,exp,msk)       _WREM16(addr,data,exp,msk)
#define WREM32(addr,data,exp,msk)       _WREM32(addr,data,exp,msk)
#define WAIT_RE8(addr,exp,i)            _WAIT_RE8(addr,exp,i)
#define WAIT_RE16(addr,exp,i)           _WAIT_RE16(addr,exp,i)
#define WAIT_RE32(addr,exp,i)           _WAIT_RE32(addr,exp,i)
#define WAIT_REM8(addr,exp,msk,i)       _WAIT_REM8(addr,exp,msk,i)
#define WAIT_REM16(addr,exp,msk,i)      _WAIT_REM16(addr,exp,msk,i)
#define WAIT_REM32(addr,exp,msk,i)      _WAIT_REM32(addr,exp,msk,i)
#define BSET8(addr,msk)                 _BSET8(addr,msk)
#define BSET16(addr,msk)                _BSET16(addr,msk)
#define BSET32(addr,msk)                _BSET32(addr,msk)
#define BCLR8(addr,msk)                 _BCLR8(addr,msk)
#define BCLR16(addr,msk)                _BCLR16(addr,msk)
#define BCLR32(addr,msk)                _BCLR32(addr,msk)
#define BINV8(addr,msk)                 _BINV8(addr,msk)
#define BINV16(addr,msk)                _BINV16(addr,msk)
#define BINV32(addr,msk)                _BINV32(addr,msk)
#define DEBUG(src,cmt)                  _DEBUG((char*)src,(char*)cmt)
#define TRACE(src,cmt)                  _TRACE((char*)src,(char*)cmt)
#define INFO(src,cmt)                   _INFO((char*)src,(char*)cmt)
#define WARN(src,cmt)                   _WARN((char*)src,(char*)cmt)
#define ERROR(src,cmt)                  _ERROR((char*)src,(char*)cmt)
#define FATAL(src,cmt)                  _FATAL((char*)src,(char*)cmt)
#define SHOW8(src,cmt,val)              _SHOW8((char*)src,(char*)cmt,val)
#define SHOW16(src,cmt,val)             _SHOW16((char*)src,(char*)cmt,val)
#define SHOW32(src,cmt,val)             _SHOW32((char*)src,(char*)cmt,val)
#endif  // (MCD_C_API == _USE_MACRO)


#endif __MCD_C_API_H    // End of "mcd_c_api.h"
